///////////////////////////////////////////////////////////
//                    ao_craft_calc                      //
//          Калькулятор крафта для Аллодов Онлайн        //
//               Copyright (C) 2014 Sagrer               //
//             Распространяется на условиях              //
//                  Modified LGPL v2.1                   //
//           см. файл COPYING.modifiedLGPL.txt           //
//                                                       //
//    http://personal.sagrer.ru/tracs/ao_craft_calc      //
///////////////////////////////////////////////////////////

//Please, use UTF8-encoding to read this file.
//Chtobi prochitat etot fail - vospolzuites kodirovkoy UTF8.

//Над данным файлом работали:
// 1) Sagrer (sagrer@yandex.ru)
// 2)

////////////////////////////////////////////////////////////////////////

///////////////////////////////////////////////////////////////
//       Набор структур и классов для работы с рецептами     //
///////////////////////////////////////////////////////////////

unit AOC_RecipesDataContainers;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils;

const
  //Константы со значениями по умолчанию.
  Def_ReagentName = 'UnknownReg';
  Def_IcoName = 'none';
  Def_CraftPrcnt = 1.0;
  Def_CritPrcnt = 1.0;
  Def_UpgradeCost = 100;
  Def_RecipeName = 'UnknownRecipe';
  Def_ServerType = 'UnknownServer';
  Def_TargetItem = 'UnknownItem';
  Def_UpgraderItem = 'UnknownItem';


type
  RReagent = record             //Структура с информацией о реагенте (секция [Reagents]).
    Name : AnsiString;   //имя реагента, в юникоде на случай если кириллица или какие-то китайские кравозяблики вдруг, мало ли.
    IcoName : AnsiString;   //имя файла с иконкой реагента, на случай если однажды будет отображаться в программе. Если содержит none - иконки нету.
  end;

  AReagents = array of RReagent;   //Тип одномерного массива для списка реагентов.

  RRecipeRegInfo = record      //Структура с информацией о влиянии реагента на рецепт (секция [RecipesRegs]).
    //RecipeName - не запоминаем т.к. записи данной структуры и так будут приписаны объекту соответствующего рецепта.
    RegName : AnsiString;   //имя реагента, должно полностью совпадать с именем в секции Reagents.
    CraftPrcnt : Double;   //количество процентов которое даёт один крафт по рецепту с этим реагентом.
    CritPrcnt : Double;   //шанс на критическое улучшение.
    UpgradeCost : LongInt;  //стоимость в голде на одну операцию улучшения. Если -1 - используется базовая стоимость из рецепта.
  end;

  ARecipeRegs = array of RRecipeRegInfo;  //Тип одномерного массива для списка записей о реагентов для рецепта.

  TRecipeInfo = class       //Класс для хранения рецепта. Сделан классом а не структурой чтобы он сам управлял памятью - там будет унутре массив для реагентов.
  private
    //Внутренние переменные
    //Закрытые методы
  protected
  public
    //Переменные
    Name : AnsiString;   // имя рецепта - нужно для юзверя чтобы понять о чём рецепт и для программы чтобы найти соответствие со строками из RecipesRegs
    ServerType : AnsiString;  //тип сервера - по сути тут будет идентификатор типа игры для которой применяется рецепт - чтобы не смешивать рецепты разных сезонов и серверов с разными правилами, мало ли вдруг кто будет считать крафт на какой нибудь пиратке ).
    TargetItem : AnsiString;  //идентификатор типа шмотки которая улучшается, например нечто обозначающее "любую рыжую шмотку 60 уровня" или "зелёный одноруч 60 уровня" или типа того. В простейшем варианте рассчётов использоваться не будет ибо не нужно но в теории эта инфа может быть полезна для рассчёта крафта в случае если один предмет можно улучшать разными типами улучшителей - тогда в теории опять же можно брать инфу из нескольких рецептов одновременно и учитывать их все исходя из наличия улучшителей у игрока.
    UpgraderItem : AnsiString;   //аналогичный предыдущему идентификатор но теперь для улучшителя.
    UpgradeCost : LongInt;   //стоимость в голде на одну операцию улучшения.
    Regs : ARecipeRegs;   //Содержимое секции [RecipesRegs] для данного рецепта.

    //Вложенные объекты

    //Конструкторы-деструкторы
    constructor Create; virtual;
    destructor Destroy; override;

    //Открытые методы
    class procedure SetDefaultsForReg(var RegLink : RRecipeRegInfo);  //Проставить значения по умолчанию реагенту в аргументе.
    procedure SetDefaultsForReg(const RegIndex : LongWord);   //Проставить значения по умолчанию реагенту с указанным индексом.
    procedure SetRegsLength(const NewLength : LongWord);   //Изменить размер массива реагентов. Если операция приведёт к увеличению массива - проставит значения по умолчанию новым элементам.
    procedure ReSortRegs();    //Пересортировать массив реагентов по их эффективности - от самого самого до самого фигового ).
    //Свойства
  end;

  ARecipes = array of TRecipeInfo;   //Тип одномерного массива для объектов с инфой о рецептах.


implementation

/////////////////////////////////////////////
//            TRecipeInfo
/////////////////////////////////////////////

//-----------------------------------------//
//        Конструкторы-деструкторы...      //
//-----------------------------------------//

constructor TRecipeInfo.Create;
begin
  //Создать вложенные объекты классов...

  //Проставить дефолтные значения...
  Self.Name := Def_RecipeName;
  Self.ServerType := Def_ServerType;
  Self.TargetItem := Def_TargetItem;
  Self.UpgraderItem := Def_UpgraderItem;
  Self.UpgradeCost := Def_UpgradeCost;

  //Массив реагентов поначалу должен быть пустым.
  SetLength(Self.Regs,0);

  //Просто забить инфу в переменные...

end;

destructor TRecipeInfo.Destroy;
begin
  //Выкидываем мусор
  SetLength(Self.Regs,0);   //В структурах нет ссылочных типов кроме строк а памятью строк паскаль рулит сам так что достаточно просто очистить массив не очищая перед этим каждый элемент руками.

  //Выполнить унаследованный деструктор
  inherited;
end;

//------------------------------------------//
//            Закрытые методы...            //
//------------------------------------------//

//------------------------------------------//
//           Защищенные методы...           //
//------------------------------------------//

//------------------------------------------//
//            Открытые методы...            //
//------------------------------------------//

class procedure TRecipeInfo.SetDefaultsForReg(var RegLink : RRecipeRegInfo);
//Проставить значения по умолчанию реагенту в аргументе.
begin
  RegLink.RegName := Def_ReagentName;
  RegLink.CraftPrcnt := Def_CraftPrcnt;
  RegLink.CritPrcnt := Def_CritPrcnt;
  RegLink.UpgradeCost := Def_UpgradeCost;
end;

procedure TRecipeInfo.SetDefaultsForReg(const RegIndex : LongWord);
//Проставить значения по умолчанию реагенту с указанным индексом.
begin
  //Тупо вбить значения элементам.
  Self.SetDefaultsForReg(Self.Regs[RegIndex]);
end;

procedure TRecipeInfo.SetRegsLength(const NewLength : LongWord);
//Изменить размер массива реагентов. Если операция приведёт к увеличению
//массива - проставит значения по умолчанию новым элементам.
var
  OldLength, i : LongWord;
begin
  //Запомним текущий размер.
  OldLength := Length(Self.Regs);
  //Меняем размер массива
  SetLength(Self.Regs,NewLength);
  //И если размер увеличился - проставляем значения по умолчанию.
  if NewLength > OldLength then begin
    for i := NewLength-1 downto OldLength do begin
      Self.SetDefaultsForReg(i);
    end;
  end;
end;

procedure TRecipeInfo.ReSortRegs();
//Пересортировать массив реагентов по их эффективности - от самого самого
//до самого фигового ).
var
  I, J, RegsCount : LongWord;
  CurrElem : RRecipeRegInfo;
  Done : Boolean;
begin
  //Что-то имеет смысл делать только если  реагентов больше одного.
  RegsCount := Length(Self.Regs);
  if RegsCount > 1 then begin
    for I := 1 to RegsCount-1 do begin
      //Если предыдущий элемент меньше текущего - работаем.
      if Self.Regs[I-1].CraftPrcnt < Self.Regs[I].CraftPrcnt then begin
        //Запоминаем текущий элемент.
        CurrElem := Self.Regs[I];
        //Затем циклично смещаем вышележащие элементы пока предыдущий элемент
        //не станет больше или равен текущему либо пока не достигнем начала массива.
        J := I;
        Done := false;
        while Done = false do begin
          if J > 0 then begin
            if Self.Regs[J-1].CraftPrcnt < CurrElem.CraftPrcnt then begin
              //Предыдущий элемент меньше текущего по I - смещаем его в текущий по J.
              Self.Regs[J] := Self.Regs[J-1];
              J := J-1;  //Движемся назад по массиву.
            end
            else begin
              //Предыдущий элемент больше или равен - втыкаем на это место запомненный
              //ранее текущий элемент и закрываем лавочку.
              Self.Regs[J] := CurrElem;
              Done := true;
            end;
          end
          else begin
            //Добрались до начала массива.
            Self.Regs[J] := CurrElem;
            Done := true;
          end;
        end;
      end;
    end;
  end;
end;

end.

