///////////////////////////////////////////////////////////
//                   ExtraFileUtilsLCL                   //
//              //
//                   LCL.            //
//          Copyright (C) 2007-2010 Gipat Group          //
//                                                       //
//                             //
//                  Modified LGPL v2.1                   //
//           .  COPYING.modifiedLGPL.txt           //
//                                                       //
//                    0.2                    //
//                                                       //
//                  www.gipatgroup.org                   //
///////////////////////////////////////////////////////////

//      , ....  :
// 1) Immortal (immortalGAD@rambler.ru)
// 2)  (Sagrer)  (sagrer@yandex.ru)

////////////////////////////////////////////////////////////////////////

unit ExtraFileUtilsLCL;

{$mode objfpc}{$H+}

interface

uses
  windows ,Classes, SysUtils, FileUtil, ExtraFunctionsLcl, syncobjs;

const
//  ,      
  FILECOPY_BUFFER_SIZE = 81920;
//   
  INVALID_SYSFILE_ATTR :LongInt = -1;
  
//  
  resOk = 0;
  resParametersError = 1;
/////////////////////////
  
type
//      .
  IOException = class(Exception) end;
  TFileSystemIterator = class;
  
////////////////////////////////////////////////////////////////////////////////
//  ,     
  TCopyFileProgressCallback = procedure(const current, total :Int64) of object;
//  ,    
  TCopyFileFinishCallback   = procedure(const resultt :Boolean) of object;

//     .
  TCopyFileThread = class (TThread)
    private
      curLock, totalLock :TCriticalSection;
      _current, _total :Int64;
    protected
      procedure Execute; override;
      function  GetTotal   :Int64;
      procedure SetTotal  (const value :Int64);
      function  GetCurrent :Int64;
      procedure SetCurrent(const value :Int64);
    public
      //  
      CopyResult :Boolean;
      CopyFileProgress :TCopyFileProgressCallback;
      CopyFileFinish   :TCopyFileFinishCallback;
      SourceFilePath   :AnsiString;
      DestFilePath     :AnsiString;
      OverwriteExists  :Boolean;  //   ,   
      property Total   :Int64 read GetTotal;
      property Current :Int64 read GetCurrent;

      constructor Create(const source, dest :AnsiString; const IsOverwriteExists , startSuspended :Boolean);
      destructor Destroy; override;
  end;

////////////////////////////////////////////////////////////////////////////////
//   ,     
  TFileSystemElemInfo = class
    protected
      SearchRec :TSearchRec;
    public
      //      
      FullPath :AnsiString;
      constructor Create(const filePath :AnsiString); virtual;
      constructor Create(const filePath: AnsiString; const searchInfo :TSearchRec); virtual;
      function  Name :AnsiString; virtual;
      function  Attributes :Longint; virtual;
      function  Exists :Boolean; virtual; abstract;
      function  CopyTo(const path :AnsiString; const overwriteExists :Boolean = false) :Boolean; virtual; abstract;
      function  Delete :Boolean; virtual ;abstract;
  end;

////////////////////////////////////////////////////////////////////////////////
//      
  TFileInfo = class (TFileSystemElemInfo)
    public
      function CopyTo(const path :AnsiString; const overwriteExists :Boolean = false) :Boolean; override;
      function Delete :Boolean; override;
      function Exists :Boolean; override;
  end;
  
////////////////////////////////////////////////////////////////////////////////
//      
  TDirectoryInfo = class (TFileSystemElemInfo)
    public
      function CopyTo(const path :AnsiString; const overwriteExists :Boolean = false) :Boolean; override;
      function Delete :Boolean; override;
      function Exists :Boolean; override;
      function GetFiles :TFileSystemIterator;
      function GetFiles(const mask :AnsiString) :TFileSystemIterator;
      function GetDirectories :TFileSystemIterator;
      function GetDirectories(const mask :AnsiString) :TFileSystemIterator;
  end;
  
////////////////////////////////////////////////////////////////////////////////
//        
  TFileSystemIterator = class
    protected
      searchInfo :TSearchRec;
      folder     :AnsiString;
    public
      Current    :TFileSystemElemInfo;  //  \
      SearchAttr :Longint;
      constructor Create(const folderPath :AnsiString; const onlyFiles :Boolean = false);
      constructor Create(const folderPath :AnsiString; const attrPredicat :Longint);
      constructor Create(const folderPath, searchMask :AnsiString; const onlyFiles :Boolean = false);
      constructor Create(const folderPath, searchMask :AnsiString; const attrPredicat :Longint); //
      destructor  Destroy; override;
      function Next :Boolean;
  end;

//      .  temp   .
  TTempFolder = class
    protected
      path :AnsiString;
      name :AnsiString;
    public
      constructor Create(const subName :AnsiString);
      destructor  Destroy; override;
      function GetTempPath :AnsiString;   //    
  end;
  
////////////////////////////////////////////////////////////////////////////////
//    
////////////////////////////////////////////////////////////////////////////////

//  
function CopyDirectory(const sourceDir, destDir :AnsiString; const overwriteExists :Boolean = false;
                        const recursive :Boolean = true) :Boolean;


//AsyncCopyFile -     
function AsyncCopyFile(const sourcePath, destPath :AnsiString; const onFinish :TCopyFileFinishCallback;
                        const overwriteExists :Boolean = false) :TThread;
function AsyncCopyFile(const sourcePath, destPath :AnsiString; const onFinish :TCopyFileFinishCallback;
                        const onProgress :TCopyFileProgressCallback; const overwriteExists :Boolean = false) :TThread;

// SysFileExists -         
function FileSystemElemExists(const path :AnsiString) :Boolean;

// IsDirectory -        "" 
function IsDirectory(const path :AnsiString) :Boolean;
function IsDirectory(const attr :Longint) :Boolean;

Function IsLocalPath(const PathStr : AnsiString) : boolean;     // true   ,  false  .       .
Function IsGlobalPath(const PathStr : AnsiString) : boolean;   // true   .    ,     (  ).
Function FixLocalPath(const PathStr : AnsiString) : AnsiString;   //        -     .
Function ResolveLocalPath(const LocalPath, GlobalPath : AnsiString) : AnsiString;  //    
Function GetLocalPath(const GlobalRoot, FullPath : AnsiString): AnsiString;    //  .
Function GetFileSize(const FileName : AnsiString) : Int64;   //  .
function GetFilesNumber(const DirName : AnsiString; const Recursive : boolean) : LongInt;//      ...
Function IsDirsChild(const DirName, ChildName : AnsiString) : boolean;   // -    ChildName  DirName.
function AddExtensionIfAbsent(const path, extension :String) :String; //      .
function GetFileNameWithoutExtension(const path :String) :String;  //   ,  
Function HasLastSlash(const Str : string) : boolean;    //     .
Function AddLastSlash(const str : string) : string;     //       ( ).
Function DelLastSlash(const Str : string) : string;     //     .


implementation


//////////////////////////  TFileSystemIterator     ///////////////////////////////
constructor TFileSystemIterator.Create(const folderPath :AnsiString; const onlyFiles :Boolean = false);
begin
  Create(folderPath, AllDirectoryEntriesMask, onlyFiles);
end;

constructor TFileSystemIterator.Create(const folderPath :AnsiString; const attrPredicat :Longint);
begin
  Create(folderPath, AllDirectoryEntriesMask, attrPredicat);
end;

constructor TFileSystemIterator.Create(const folderPath, searchMask :AnsiString; const onlyFiles :Boolean = false);
var
  temp :Longint;
begin
  temp := faAnyFile;
  if onlyFiles then begin
    temp := temp and not faDirectory;
  end;
  Create(folderPath, searchMask, temp);
end;

constructor TFileSystemIterator.Create(const folderPath, searchMask :AnsiString; const attrPredicat :Longint);
begin
  inherited Create;
  Self.SearchAttr := attrPredicat;
  Self.folder := folderPath;
  FindFirst(AppendPathDelim(folderPath) + searchMask, faAnyFile, searchInfo);
end;

destructor TFileSystemIterator.Destroy;
begin
  inherited;
  FindClose(searchInfo);
end;

function TFileSystemIterator.Next :Boolean;
begin
  Result := (FindNext(searchInfo) = 0);
  if not Result then exit;
  
  if (searchInfo.Name = '.') or (searchInfo.Name = '..') then begin
    Result := Self.Next;
    exit;
  end;
  
  if (searchInfo.Attr and SearchAttr) = 0 then begin
    Result := Self.Next;
    exit;
  end;

  if IsDirectory(searchInfo.Attr) then
    Current := TDirectoryInfo.Create(folder, searchInfo)
  else
    Current := TFileInfo.Create(folder, searchInfo);

end;

///////////////////              TCopyFileThread             ///////////////////

//  
constructor TCopyFileThread.Create(const source, dest :AnsiString; const IsOverwriteExists, startSuspended :Boolean);
begin
  inherited Create(startSuspended);
  SourceFilePath := source;
  DestFilePath   := dest;
  self.OverwriteExists := IsOverwriteExists;
  self._current := 0;
  Self._total := -1;
  totalLock := TCriticalSection.Create;
  curLock   := TCriticalSection.Create;
end;

destructor TCopyFileThread.Destroy;
begin
  inherited;
  totalLock.Free;
  curLock.Free;
end;

//   ,  .
procedure TCopyFileThread.Execute;
var
  sourceStream :TFileStream;
  destStream   :TFileStream;
  curr, max    :Int64;
  stepReaded   :Integer;
  part, temp   :Int64;
  buffer       :array[0 .. FILECOPY_BUFFER_SIZE - 1] of byte;
begin
  CopyResult := false;
  
  if not FileExists(SourceFilePath) then begin
    if assigned (CopyFileFinish) then CopyFileFinish(false);
    exit;
  end;
  
  //        .   -     ,
  //    .
  if FileExists(DestFilePath) then begin
    if (not OverwriteExists) or (not DeleteFile(DestFilePath)) then begin
      if assigned (CopyFileFinish) then CopyFileFinish(false);
      exit;
    end;
  end;
  
  //  
  sourceStream := TFileStream.Create(SourceFilePath, fmOpenRead);
  destStream   := TFileStream.Create(DestFilePath, fmCreate);
  
  curr := 0;
  max := sourceStream.Size;
  SetTotal(max);
  part := total div 100;
  temp := 0;
  buffer[0] := 0;

  try
    while (curr < max) do begin
      stepReaded := sourceStream.Read(buffer, FILECOPY_BUFFER_SIZE);
      destStream.Write(buffer, stepReaded);
      curr := curr + stepReaded;
      SetCurrent(curr);
      //         100- ,
      //     .
      if assigned(CopyFileProgress) then begin
        temp := temp + stepReaded;
        if temp >= part then begin
          temp := temp mod part;
          CopyFileProgress(curr, max);
        end;
      end;
    end;
  finally
    sourceStream.Free;
    destStream.Free;
  end;
  
  CopyResult := true;
  if assigned (CopyFileFinish) then CopyFileFinish(CopyResult);

end;


function TCopyFileThread.GetTotal :Int64;
begin
  totalLock.Acquire;
  try
    Result := _total;
  finally
    totalLock.Release;
  end;
end;

procedure TCopyFileThread.SetTotal(const value :Int64);
begin
  totalLock.Acquire;
  try
    _total := value;
  finally
    totalLock.Release;
  end;
end;

function TCopyFileThread.GetCurrent :Int64;
begin
  curLock.Acquire;
  try
    Result := _current;
  finally
    curLock.Release;
  end;
end;

procedure TCopyFileThread.SetCurrent(const value :Int64);
begin
  curLock.Acquire;
  try
    _current := value;
  finally
    curLock.Release;
  end;
end;

///////////////////////  TFileSystemIterator  & co    /////////////////////////////

// 
constructor TFileSystemElemInfo.Create(const filePath :AnsiString);
begin
  inherited Create;
  Self.FullPath := filePath;
end;

// 
constructor TFileSystemElemInfo.Create(const filePath: AnsiString; const searchInfo :TSearchRec);
begin
  Create(AppendPathDelim(filePath) + searchInfo.Name);
  Self.SearchRec := searchInfo;
end;

//  
function TFileSystemElemInfo.Attributes :Longint;
begin
  //    SearchRec
  if (SearchRec.FindHandle = 0) then
    Result := INVALID_SYSFILE_ATTR
  else
    Result := SearchRec.Attr;
end;

//    
function TFileSystemElemInfo.Name :AnsiString;
begin
  Result := SearchRec.Name;
  if (Result = '') then
    Result := ExtractFileName(FullPath);
end;

// 
function TFileInfo.CopyTo(const path :AnsiString; const overwriteExists :Boolean = false) :Boolean;
begin
  if FileExists(path) then begin
    if overwriteExists then begin
      if not DeleteFile(path) then begin
        Result := false;
        exit;
      end;
    end else begin
      Result := true;
      exit;
    end;
  end;
  Result := CopyFile(FullPath, path);
end;

// 
function TFileInfo.Delete :Boolean;
begin
  Result := false;
  if FileExists(FullPath) then
    Result := DeleteFile(FullPath);
end;

//       ?
function TFileInfo.Exists :Boolean;
begin
  Result := FileExists(FullPath);
end;

//  .
function TDirectoryInfo.CopyTo(const path :AnsiString; const overwriteExists :Boolean = false) :Boolean;
begin
  Result := CopyDirectory(FullPath, path, overwriteExists, true);
end;

//  .
function TDirectoryInfo.Delete :Boolean;
begin
  Result := DeleteDirectory(FullPath, false);
end;

//   
function TDirectoryInfo.Exists :Boolean;
begin
  Result := DirectoryExists(FullPath);
end;

//      
function TDirectoryInfo.GetFiles: TFileSystemIterator;
begin
  Result := GetFiles('*');
end;

//     -
function TDirectoryInfo.GetFiles(const mask :AnsiString) :TFileSystemIterator;
begin
  Result := TFileSystemIterator.Create(FullPath, mask, true);
end;

function TDirectoryInfo.GetDirectories :TFileSystemIterator;
begin
  Result := GetDirectories('*');
end;

function TDirectoryInfo.GetDirectories(const mask :AnsiString) :TFileSystemIterator;
begin
  Result := TFileSystemIterator.Create(FullPath, mask, faDirectory);
end;

//////////////////////////  TTempFolder     ///////////////////////////////
// .   
constructor TTempFolder.Create(const subName :AnsiString);
var
  temp :TGUID;
begin
  CreateGUID(temp);
  //    
  path := AppendPathDelim(GetTempDir) + GUIDToString(temp) + PathDelim;
  ForceDirectory(path + subName);
  name := subName;
end;

//   
destructor  TTempFolder.Destroy;
begin
  inherited;
  DeleteDirectory(path, false);
end;

//    
function TTempFolder.GetTempPath :AnsiString;
begin
  Result := AppendPathDelim(path + name);
end;

////////////////////////////////////////////////////////////////////////////////
//  
function CopyDirectory(const sourceDir, destDir :AnsiString; const overwriteExists :Boolean = false;
                        const recursive :Boolean = true) :Boolean;
var
  it :TFileSystemIterator;
begin
  Result := false;
  if not DirectoryExists(sourceDir) then begin
    exit;
  end;
  //  destDir   , 
  if not DirectoryExists(destDir) then ForceDirectory(destDir);
  it := TFileSystemIterator.Create(sourceDir, not recursive);
  try
    while it.Next do begin
      if not it.Current.CopyTo(AppendPathDelim(destDir) + it.Current.Name, overwriteExists) then begin
        exit;
      end;
    end;
    Result := true;
  finally
    it.Free;
  end;
end;


//     
function AsyncCopyFile(const sourcePath, destPath :AnsiString; const onFinish :TCopyFileFinishCallback;
                        const overwriteExists :Boolean = false) :TThread;
begin
  Result := AsyncCopyFile(sourcePath, destPath, onFinish, nil, overwriteExists);
end;

//     
function AsyncCopyFile(const sourcePath, destPath :AnsiString; const onFinish :TCopyFileFinishCallback;
                       const onProgress :TCopyFileProgressCallback; const overwriteExists :Boolean = false) :TThread;
var
  worker :TCopyFileThread;
begin
  worker := TCopyFileThread.Create(sourcePath, destPath, overwriteExists ,true);
  worker.CopyFileProgress := onProgress;
  worker.CopyFileFinish   := onFinish;
  worker.FreeOnTerminate  := true;
  worker.Resume;
  Result := worker;
end;

function FileSystemElemExists(const path :AnsiString) :Boolean;
begin
  Result := (FileGetAttr(path) <> INVALID_SYSFILE_ATTR);
end;

//       ?
function IsDirectory(const path :AnsiString) :Boolean;
var
  attr :Longint;
begin
  attr := FileGetAttr(path);
  if attr <> INVALID_SYSFILE_ATTR then begin
    Result := IsDirectory(attr);
  end else
    Result  := false;
end;

//        
function IsDirectory(const attr :Longint) :Boolean;
begin
  Result := (attr and faDirectory) = faDirectory;
end;


////////////////     ExtraFunctionsCLC  ////////////////////////////////
Function IsLocalPath(const PathStr : string) : boolean;
// true   ,  false  .
//      .
begin
  Result := false;   //     .

  if Length(PathStr) >= 1 then begin
    if PathStr[1] = '.' then begin
      Result := true;
    end;
  end;
end;

Function IsGlobalPath(const PathStr : AnsiString) : boolean;
// true   .    ,
//    (  ).
begin
  Result := false;   //      .

  if Length(PathStr) >= 3 then begin
    if PathStr[2]+PathStr[3] = ':\' then begin
      Result := true;
    end;
  end;
end;

Function FixLocalPath(const PathStr : AnsiString) : AnsiString;
//      
// -     .
begin
  Result := PathStr;

  if IsLocalPath(Result) = false then begin
    if Length(Result) >= 1 then begin
      if Result[1] <> '\' then begin
        Result := '\'+Result;
      end;
    end
    else begin
      Result := '\'+Result;
    end;
    Result := '.'+Result;
  end;
end;

Function ResolveLocalPath(const LocalPath, GlobalPath : AnsiString) : AnsiString;
//    .      - 
// .        -   
//  .
var
  TempLocalPath, LocalParsed : AnsiString;
begin
  // :        "".
  //    ,   ,
  //     ( ) -   -   ,
  //  -        ,
  // -  -         .
  //    .

  //    ,   .
  Result := ChompPathDelim(GlobalPath);
  TempLocalPath := LocalPath;     //       .
  
  // -     ?    -    .
  if IsGlobalPath(TempLocalPath) = false then begin
    //  .     ?
    if IsLocalPath(TempLocalPath) = false then begin
      //     - .
      TempLocalPath := FixLocalPath(TempLocalPath);
    end;
  end;
  
  // -      -  .
  if IsLocalPath(TempLocalPath) = true then begin
    repeat
      //...
      LocalParsed := Parse(TempLocalPath,'\');

      //  ...
      if (LocalParsed = '.') or (LocalParsed = '') then begin
        //       -    .
      end
      else if LocalParsed = '..' then begin
        //  -    ...
        ParseTail(Result,'\');
      end
      else begin
        //   -  -     ...
        Result := Result+'\'+LocalParsed;
      end;

    until TempLocalPath = '';

    //.    LocalPath     -    ...
    if HasLastSlash(LocalPath) = true then begin
      Result := Result+'\';
    end;
  end
  else begin
    //      -   .
    Result := TempLocalPath;
  end;

end;

Function GetLocalPath(const GlobalRoot, FullPath : AnsiString): AnsiString;
//  .
var
  TmpGlobalRoot, TmpFullPath, CommonPart, TempStr : AnsiString;
  Finded : boolean;

begin
  //         .
  if StrPos(PChar(UpperCase(FullPath)),PChar(DelLastSlash(UpperCase(GlobalRoot)))) = nil then begin

    // -         -      ,
    //        .
    TmpGlobalRoot := AddLastSlash(GlobalRoot);
    TmpFullPath := AddLastSlash(FullPath);
    if UpperCase(Parse(TmpGlobalRoot,'\')) = UpperCase(Parse(TmpFullPath,'\')) then begin
      // ""     -      "".

      //   ...
      CommonPart := '';
      Finded := false;
      TmpGlobalRoot := DelLastSlash(GlobalRoot);
      TmpFullPath := DelLastSlash(FullPath);
      repeat
        TempStr := Parse(TmpGlobalRoot,'\');
        if UpperCase(TempStr) = UpperCase(Parse(TmpFullPath,'\')) then begin
          //  .
          CommonPart := CommonPart+TempStr+'\';
        end
        else begin
          // .  .     .
          Finded := true;
        end;
      until Finded = true;

      //        ...
      TmpGlobalRoot := AddLastSlash(GlobalRoot);
      TmpFullPath := AddLastSlash(FullPath);
      TmpGlobalRoot := DelLastSlash(RightStr(TmpGlobalRoot,Length(TmpGlobalRoot)-Length(CommonPart)));
      TmpFullPath := DelLastSlash(RightStr(TmpFullPath,Length(TmpFullPath)-Length(CommonPart)));
      //      ...
      Result := '';
      repeat
        //  TmpGlobalRoot  .
        if Parse(TmpGlobalRoot,'\') <> '' then begin
          //  -  -   ...
          Result := Result+'..\';
        end;
      until TmpGlobalRoot = '';
      //    .
      Result := Result+TmpFullPath;
      //,  .
      Exit;
    end
    else begin
      // ""-  -    
      Result := FullPath;
      Exit;
    end;

  end;

  //            -    ./
  if UpperCase(DelLastSlash(GlobalRoot)) = UpperCase(DelLastSlash(FullPath)) then begin
    Result := '.\';
    Exit;
  end;

  //   ...
  TmpGlobalRoot := AddLastSlash(GlobalRoot);
  TmpFullPath := FullPath;

  TmpFullPath := RightStr(TmpFullPath,Length(TmpFullPath)-Length(TmpGlobalRoot));
  Result := '.\'+TmpFullPath;
end;

Function GetFileSize(const FileName : AnsiString) : Int64;
//  .
var
  TheFileStream : TFileStream;
begin
  //   .
  Result := 0;
  //  ...
  if FileExists(FileName) = true then begin
    //...   ...
    TheFileStream := TFileStream.Create(FileName,fmOpenRead);
    //...   .
    Result := TheFileStream.Size;
    //  .
    TheFileStream.Free;
  end;
end;

Function GetFilesNumber(const DirName : AnsiString; const Recursive : boolean) : LongInt;
//      ...
var
  SearchRec : TSearchRec;
begin
  Result := 0;

  if FindFirst(AddLastSlash(DirName)+'*', faAnyFile and faDirectory, SearchRec) = 0 then begin
    //    -  (  )  .
    repeat
      if (SearchRec.Name <> '.') and (SearchRec.Name <> '..') then begin
        //- .
        Result := Result+1;
        //Writeln(SearchRec.Name);
        if Recursive = true then begin
          //   ...
          if (SearchRec.Attr and faDirectory) = faDirectory then begin
            //...     -    .
            Result := Result+GetFilesNumber(AddLastSlash(DirName)+SearchRec.Name, Recursive);
          end;
        end;
      end;;
    until FindNext(SearchRec) <> 0;
    SysUtils.FindClose(SearchRec);
  end;
  //Writeln(IntToStr(Result));

end;

Function IsDirsChild(const DirName, ChildName : AnsiString) : boolean;
// -    ChildName  DirName.
var
  TempStr : AnsiString;
begin
  //.
  Result := false;     //    child-

  // " ".
  TempStr := GetLocalPath(DirName,ChildName);
  // -    ".\"
  if Length(TempStr) > 1 then begin
    //      .
    if TempStr[1]+TempStr[2] = '.\' then begin
      //   child- DirName
      Result := true;
    end;
  end;
end;

//      .
function AddExtensionIfAbsent(const path, extension :String) :String;
var
  ex :AnsiString;
begin
  Result := path;
  ex := ExtractFileExt(path);
  if (ex = '') then begin
    Result := path + extension;
  end;
end;

//   ,  
function GetFileNameWithoutExtension(const path :String) :String;
var
  fileName :String;
  i        :Integer;
begin
  fileName := ExtractFileName(path);
  i := Length(FileName);
  while (i > 0) and (FileName[i] <> '.') do Dec(i);

  if (i > 0) then
    Result := Copy(FileName, 1, i - 1)
  else
    Result := fileName;
end;

Function HasLastSlash(const Str : string) : boolean;
//     .
var
  I : integer;
  LastCh : char;
begin
  if Length(Str) > 0 then begin
    // -       .
    I := 0;
    LastCh := #0;
    repeat
      I := I+1;
      If str[I] <> #0 then begin
        LastCh := str[I];
      end;
    until str[I] = #0;
    //      ,  :
    If LastCh = '\' then result := true
    else result := false;
  end
  else begin
    //    -    .
    Result := false;
  end;
end;

Function AddLastSlash(const str : string): string;
//       ( ).
begin
  //     -    .
  If HasLastSlash(str) = false then result := str+'\'
  else result := str;
end;

Function DelLastSlash(const Str : string) : string;
//     .
begin
  Result := '';
  if HasLastSlash(Str) = true then begin
    SetLength(Result,Length(Str)-1);
    CopyMemory(@Result[1],@Str[1],Length(Str)-1);
  end
  else begin
    Result := Str;
  end;
end;

end.

