///////////////////////////////////////////////////////////
//                    ao_craft_calc                      //
//          Калькулятор крафта для Аллодов Онлайн        //
//               Copyright (C) 2014 Sagrer               //
//             Распространяется на условиях              //
//                  Modified LGPL v2.1                   //
//           см. файл COPYING.modifiedLGPL.txt           //
//                                                       //
//    http://personal.sagrer.ru/tracs/ao_craft_calc      //
///////////////////////////////////////////////////////////

//Please, use UTF8-encoding to read this file.
//Chtobi prochitat etot fail - vospolzuites kodirovkoy UTF8.

//Над данным файлом работали:
// 1) Sagrer (sagrer@yandex.ru)
// 2)

////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////
//       Класс объекта, собственно и вычисляющего оптимальный крафт     //
//////////////////////////////////////////////////////////////////////////

unit AOC_CraftCalcEngine;

{$mode objfpc}{$H+}

interface

uses
  Classes, SysUtils, AOC_CraftCalcResult, AOC_RecipesDataContainers;

type
  TCraftCalcEngine = class
  private
    //Внутренние переменные
    //Закрытые методы
  protected
  public
    //Переменные
    ActiveRecipe : TRecipeInfo;    //Ссылка на созданный в другом месте объект рецепта. Нельзя удалять\создавать внутри этого объекта.
    Results : ACraftCalcResults;   //Массив для результатов.
    MaxRegsCount : array of LongInt;  //Массив для максимального количества подбираемых реагентов, порядок тот же что в рецепте.
    AlreadyDone : double;   //Сколько процентов уже прокрафчено.
    BadResultsCount : LongInt;   //Сколько результатов помечено как плохие.
    //Вложенные объекты

    //Конструкторы-деструкторы
    constructor Create; virtual;
    destructor Destroy; override;

    //Открытые методы
    procedure SetResultsLength(const NewLength : LongWord);   //Изменить размер массива результатов. Если операция приведёт к увеличению массива - создаст новые объекты, иначе удалит.
    procedure MakeTheCalculation();            //Подбирает варианты крафта исходя из забитых в объект параметров, результат соответственно падает в Results.
    procedure Clear();                         //Метод для очистки содержимого объекта - чтобы не пересоздавать его каждый раз после использования.
    procedure SortResults();                   //Метод сортирует результаты по количеству нужных улучшителей.
    procedure CheckForBadResults();            //Проверка результатов и пометка неудачных вариантов чтобы не отображать их юзверю.
    //Свойства
  end;


implementation

uses math;

/////////////////////////////////////////////
//            TCraftCalcEngine
/////////////////////////////////////////////

//-----------------------------------------//
//        Конструкторы-деструкторы...      //
//-----------------------------------------//

constructor TCraftCalcEngine.Create;
begin
  //Создать вложенные объекты классов...

  //Проставить дефолтные значения...
  Self.Clear();

  //Просто забить инфу в переменные...

end;

destructor TCraftCalcEngine.Destroy;
begin
  //Выкидываем мусор
  Self.SetResultsLength(0);
  SetLength(MaxRegsCount,0);

  //Выполнить унаследованный деструктор
  inherited;
end;

//------------------------------------------//
//            Закрытые методы...            //
//------------------------------------------//

//------------------------------------------//
//           Защищенные методы...           //
//------------------------------------------//

//------------------------------------------//
//            Открытые методы...            //
//------------------------------------------//

procedure TCraftCalcEngine.SetResultsLength(const NewLength : LongWord);
//Изменить размер массива результатов. Если операция приведёт к увеличению
//массива - создаст новые объекты, иначе удалит.
var
  OldLength, i : LongWord;
begin
  //Запомним текущий размер.
  OldLength := Length(Self.Results);
  //И если размер увеличился - создаём объекты.
  if NewLength > OldLength then begin
    //Меняем размер массива
    SetLength(Self.Results,NewLength);
    for i := NewLength-1 downto OldLength do begin
      Self.Results[i] := TCraftCalcResult.Create;
    end;
  end
  else if OldLength > NewLength then begin
    //Если размер массива уменьшается - сначала надо очистить удаляемые элементы.
    for i := OldLength-1 downto NewLength do begin
      Self.Results[i].Free;
    end;
    //А теперь можно и уменьшить размер массива.
    SetLength(Self.Results,NewLength);
  end;
end;

procedure TCraftCalcEngine.MakeTheCalculation();
//Подбирает варианты крафта исходя из забитых в объект параметров, результат
//соответственно падает в Results.
var
  RegsCount, MaxRegsNeeded, I, J, CurrCraftCost, CurrItemsNum, CurrResultsNum : LongInt;
  TempFloat, CurrCraftPower, NeededPower  : Double;
  AllOk, CalcFinished : Boolean;
  CurrRegsCount : array of LongInt;

begin
  //Инициализация
  AllOk := true;

  //Имеет смысл что-то делать только если в рецепте есть реагенты и указан процент
  //прокрафта меньше сотни )).
  RegsCount := Length(Self.ActiveRecipe.Regs);
  if (RegsCount > 0) and (Self.AlreadyDone < 100) then begin
    //Вычисляем сколько максимум может потребоваться улучшений.
    NeededPower := 100-Self.AlreadyDone;
    TempFloat := NeededPower / Self.ActiveRecipe.Regs[RegsCount-1].CraftPrcnt;
    MaxRegsNeeded := ceil(TempFloat);     //Округляем до бОльшего.
  end
  else AllOk := false;

  //Если всё ок - заменим нолики на максимально имеющее смысл количество реагентов.
  //Минус один же - заменяем на нолики ).
  if AllOk = true then begin
    for I := 0 to RegsCount-1 do begin
      if Self.MaxRegsCount[I] = 0 then Self.MaxRegsCount[I] := MaxRegsNeeded
      else if Self.MaxRegsCount[I] = -1 then Self.MaxRegsCount[I] := 0;
    end;

    //А теперь самая интересная часть ))). Ломимся в какбэ бесконечном цикле по
    //"разрядам" какбэ числа где каждый разряд - количество одного из типов
    //реагентов а максимальное значение разряда - максимальное количество этого
    //реагента. Ну и просто этому какбэ числу прибавляем по единичке и считаем
    //что даст эта комбинация реагентов, если оно нас устраивает - запоминаем,
    //иначе просто ломимся дальше, всё просто же. I у нас будет номером текущего
    //разряда ).

    //Выставим начальные значения переменных.
    CalcFinished := false;
    SetLength(CurrRegsCount,RegsCount);      //Массив для текущих подбираемых значений количества.
    //Заполним массив нулями.
    for I := 0 to RegsCount-1 do begin
      CurrRegsCount[I] := 0;
    end;
    //Теперь собсно бегающий по разрядам цикл.
    I := RegsCount-1;      //Начинать с самого слабого реагента.
    while CalcFinished = false do begin
      //+1 однако )).
      CurrRegsCount[I] := CurrRegsCount[I]+1;
      //А теперь смотрим что получилось.
      if CurrRegsCount[I] > Self.MaxRegsCount[I] then begin
        //Число превысило максимум - обнуляем его.
        CurrRegsCount[I] := 0;
        //И смотрим есть ли выше разряды.
        if I > 0 then begin
          //Нам всё ещё есть куда шевелиться - уменьшаем счётчик разрядов,
          //дальше работать будет уже следующая итерация.
          I := I-1;
        end
        else begin
          //Всё, дошли до упора, дальше считать некуда - можно вываливаться
          //из цикла. По идее эта ветка должна быть единственным выходом из while.
          CalcFinished := true;
        end;
      end
      else begin
        //Число где то в нормальных рамках - смотрим какой сейчас активен разряд.
        if I <> RegsCount-1 then begin
          //Разряд не для самого слабого реагента - переходим на разряд
          //этого самого слабого и го в следующую итерацию. Нолик там тоже надо
          //считать поэтому поставим там -1.
          I := RegsCount-1;
          CurrRegsCount[I] := -1;
        end
        else begin
          //А вот и собсно самая главная ветка цикла - мы сейчас на
          //разряде самого слабого реагента и все числа в допустимых
          //пределах - т.е. реакция возможна, надо посчитать что по ней
          //получится.
          CurrCraftPower := 0;
          CurrItemsNum := 0;
          CurrCraftCost := 0;
          for J := 0 to RegsCount-1 do begin
            //Процент прокрафта
            CurrCraftPower := CurrCraftPower+(Self.ActiveRecipe.Regs[J].CraftPrcnt*CurrRegsCount[J]);
            //Количество крафтов.
            CurrItemsNum := CurrItemsNum+CurrRegsCount[J];
            //Стоимость.
            if Self.ActiveRecipe.Regs[J].UpgradeCost = -1 then begin
              //Считаем по стоимости рецепта.
              CurrCraftCost := CurrCraftCost+(Self.ActiveRecipe.UpgradeCost*CurrRegsCount[J]);
            end
            else begin
              //Считаем по стоимости из реагента.
              CurrCraftCost := CurrCraftCost+(Self.ActiveRecipe.Regs[J].UpgradeCost*CurrRegsCount[J]);
            end;
          end;
          //Посчитали. Теперь смотрим устраивает ли нас результат.
          //Проверяем - набралось ли 100+% и не слишком ли много получилось
          //т.е. допустимо не больше 1 крафта самым слабым реагентом.
          if (CurrCraftPower >= NeededPower) and ((CurrCraftPower-NeededPower) < Self.ActiveRecipe.Regs[RegsCount-1].CraftPrcnt) then begin
            //Вроде бы всё ок. Запоминаем этот результат. Добавляем новую "строчку"...
            CurrResultsNum := Length(Self.Results)+1;
            Self.SetResultsLength(CurrResultsNum);
            //Выделяем там в массиве элементы под количество регов
            SetLength(Self.Results[CurrResultsNum-1].RegsCount,RegsCount);
            //Пишем количество.
            for J := 0 to RegsCount-1 do begin
              Self.Results[CurrResultsNum-1].RegsCount[J] := CurrRegsCount[J];
            end;
            //Стоимость.
            Self.Results[CurrResultsNum-1].CraftCost := CurrCraftCost;
            //Потеря процентов.
            Self.Results[CurrResultsNum-1].LostPower := CurrCraftPower-NeededPower;
            //Количество улучшителей.
            Self.Results[CurrResultsNum-1].ItemsNum := CurrItemsNum;
            //Ну и, собсно, вот. Как то так ).
          end;
        end;
      end;
    end;

    //Результаты получены - надо отсортировать.
    Self.SortResults();
    //И пометим плохие варианты как, собсно, плохие ).
    Self.CheckForBadResults();
  end;
end;

procedure TCraftCalcEngine.Clear();
//Метод для очистки содержимого объекта - чтобы не пересоздавать его каждый раз
//после использования.
begin
  Self.SetResultsLength(0);
  SetLength(MaxRegsCount,0);
  ActiveRecipe := nil;
  AlreadyDone := 0;
  BadResultsCount := 0;
end;

procedure TCraftCalcEngine.SortResults();
//Метод сортирует результаты по количеству нужных улучшителей.
var
  I, J, ResultsCount : LongWord;
  CurrElem : TCraftCalcResult;
  Done : Boolean;
begin
  //Что-то имеет смысл делать только если результатов больше одного.
  ResultsCount := Length(Self.Results);
  if ResultsCount > 1 then begin
    for I := 1 to ResultsCount-1 do begin
      //Если предыдущий элемент больше текущего - работаем.
      if Self.Results[I-1].ItemsNum > Self.Results[I].ItemsNum then begin
        //Запоминаем текущий элемент.
        CurrElem := Self.Results[I];
        //Затем циклично смещаем вышележащие элементы пока предыдущий элемент
        //не станет больше или равен текущему либо пока не достигнем начала массива.
        J := I;
        Done := false;
        while Done = false do begin
          if J > 0 then begin
            if Self.Results[J-1].ItemsNum > CurrElem.ItemsNum then begin
              //Предыдущий элемент меньше текущего по I - смещаем его в текущий по J.
              Self.Results[J] := Self.Results[J-1];
              J := J-1;  //Движемся назад по массиву.
            end
            else begin
              //Предыдущий элемент больше или равен - втыкаем на это место запомненный
              //ранее текущий элемент и закрываем лавочку.
              Self.Results[J] := CurrElem;
              Done := true;
            end;
          end
          else begin
            //Добрались до начала массива.
            Self.Results[J] := CurrElem;
            Done := true;
          end;
        end;
      end;
    end;
  end;
end;

procedure TCraftCalcEngine.CheckForBadResults();
//Проверка результатов и пометка неудачных вариантов чтобы не отображать
//их юзверю.
var
  I, K, L, M, ResultsCount, RegsCount, VariantsCount : LongInt;
  VariantsList : ALongInts2D;
  FindedVaredResult, IsEqual : Boolean;

begin
  //Если результаты вообще есть (и больше одного) и реагентов больше одного - поехали ).
  ResultsCount := Length(Self.Results);
  RegsCount := Length(Self.ActiveRecipe.Regs);
  Self.BadResultsCount := 0;
  if (ResultsCount > 1) and (RegsCount > 1) then begin
    //Едем собсно по массиву результатов.
    for I := 0 to ResultsCount-1 do begin
      //Тут будет аццкая вложенность циклов, что означают счётчики:
      //I - текущий рецепт.
      //J - хз, походу я его куда то пропил.
      //K - текущий проверяемый на -1 реагент рецепта.
      //L - текущий проверяемый на +1 реагент рецепта.
      //M - текущий переписываемый в список вариантов реагент рецепта.

      //Чистим список вариантов.
      SetLength(VariantsList,0,0);
      //Проходимся по реагентам результата за исключением последнего который требует минимум ресов.
      for K := 0 to RegsCount-2 do begin
        //Если количество текущего реагента не 0 - формируем варианты крафта в котором этого
        //реагента на 1 меньше а любого ниже качеством на 1 больше...
        if Self.Results[I].RegsCount[K] <> 0 then begin
          //Едем по реагентам после текущего.
          for L := K+1 to RegsCount-1 do begin
            //Что-то вообще делать имеет смысл только если +1 реагент меньше максимума.
            if Self.Results[I].RegsCount[L] < Self.MaxRegsCount[L] then begin
              //Добавляем вариант.
              VariantsCount := Length(VariantsList)+1;
              SetLength(VariantsList,VariantsCount,RegsCount);
              //Копируем количество реагентов с нужными изменениями
              for M := 0 to RegsCount-1 do begin
                if M < K then begin
                  //Реагенты до текущего копируем без изменений.
                  VariantsList[VariantsCount-1,M] := Self.Results[I].RegsCount[M];
                end
                else if M = K then begin
                  //Текущий реагент берём с -1.
                  VariantsList[VariantsCount-1,M] := Self.Results[I].RegsCount[M]-1;
                end
                else if (M > K) and (M = L) then begin
                  //+1 реагент (тот который проверяем на +1) берём с +1
                  VariantsList[VariantsCount-1,M] := Self.Results[I].RegsCount[M]+1;
                end
                else if M > K then begin
                  //все остальные последующие реагенты берём опять же без изменений.
                  VariantsList[VariantsCount-1,M] := Self.Results[I].RegsCount[M];
                end;
              end;   //for M := 0 to RegsCount-1 do begin
            end;   //if Self.Results[I].RegsCount[L] < Self.MaxRegsCount[L] then begin
          end;   //for L := K+1 to RegsCount-1 do begin
        end;   //if Self.Results[I].RegsCount[K] <> 0 then begin
      end;   //for K := 0 to RegsCount-2 do begin

      //Если какие-то лучшие варианты вообще могли существовать - их список сейчас
      //должен быть сформирован. Смотрим, если что-то есть - проверим это дело
      //на совпадение по результатам.
      VariantsCount := Length(VariantsList);
      if VariantsCount > 0 then begin
        //Что-то есть. K у нас теперь будет текушим проверяемым результатом -
        //собсно и проедемся заново по списку результатов.
        FindedVaredResult := false;   //Индикатор что нашлось совпадение - чтобы вовремя прервать циклы.
        for K := 0 to ResultsCount-1 do begin
          //Сверяем. L у нас теперь будет текущий проверяемый вариант.
          for L := 0 to VariantsCount-1 do begin
            //M - текущий сверяемый реагент ))).
            IsEqual := true;    //При несовпадении тут сразу станет false.
            for M := 0 to RegsCount-1 do begin
              if Self.Results[K].RegsCount[M] <> VariantsList[L,M] then begin
                //Несовпадение.
                IsEqual := false;
                Break;
              end;
            end;
            //Смотрим не совпало ли.
            if IsEqual = true then begin
              //Есть совпадение - значит результат у нас фиговый, надо его пометить
              //и выйти из циклов.
              Self.Results[I].IsBadResult := true;
              Self.BadResultsCount := Self.BadResultsCount+1;    //Это количество может пригодиться юзверю класса.
              FindedVaredResult := true;
              Break;
            end;
          end;

          //Если было найдено совпадение - выпасть из цикла.
          if FindedVaredResult = true then Break;
        end;
      end;
    end;
  end;
end;

end.

